// ========================
// Variables
// ========================
const popoverTriggers = document.querySelectorAll('.popover-trigger')

// ========================
// Functions
// ========================
/**
 * Generates a unique string
 * @param {Number} length
 */
const generateUniqueString = length => {
  return Math.random().toString(36).substring(2, 2 + length)
}

/**
 * Finds a popover from the trigger
 * @param {HTMLElement} popoverTrigger
 */
const getPopover = popoverTrigger => {
  return document.querySelector(`#${popoverTrigger.dataset.target}`)
}

/**
 * Finds the trigger of a popover
 * @param {HTMLElement} popover
 */
const getPopoverTrigger = popover => {
  return document.querySelector(`.popover-trigger[data-target="${popover.id}"]`)
}

/**
 * Creates a popover according to the trigger
 * @param {HTMLElement} popoverTrigger
 * @returns {HTMLElement}
 */
const createPopover = popoverTrigger => {
  const popover = document.createElement('div')
  popover.classList.add('popover')
  popover.dataset.position = popoverTrigger.dataset.popoverPosition

  // Dynamic id
  const id = generateUniqueString(5)
  popover.id = id
  popoverTrigger.dataset.target = id

  const p = document.createElement('p')
  p.textContent = popoverTrigger.dataset.content

  popover.appendChild(p)
  document.body.appendChild(popover)
  return popover
}

/**
 * Calculates top and left position of popover
 * @param {HTMLElement} popoverTrigger
 * @param {HTMLElement} popover
 * @returns {Object} Top and left values in px (without units)
 */
const calculatePopoverPosition = (popoverTrigger, popover) => {
  const popoverTriggerRect = popoverTrigger.getBoundingClientRect()
  const popoverRect = popover.getBoundingClientRect()
  const { position } = popover.dataset
  const space = 20

  if (position === 'top') {
    return {
      left: (popoverTriggerRect.left + popoverTriggerRect.right) / 2 - popoverRect.width / 2,
      top: popoverTriggerRect.top - popoverRect.height - space
    }
  }

  if (position === 'left') {
    return {
      left: popoverTriggerRect.left - popoverRect.width - space,
      top: (popoverTriggerRect.top + popoverTriggerRect.bottom) / 2 -
      (popoverRect.height / 2)
    }
  }

  if (position === 'right') {
    return {
      left: popoverTriggerRect.right + space,
      top: (popoverTriggerRect.top + popoverTriggerRect.bottom) / 2 - popoverRect.height / 2
    }
  }

  if (position === 'bottom') {
    return {
      left: (popoverTriggerRect.left + popoverTriggerRect.right) / 2 - popoverRect.width / 2,
      top: popoverTriggerRect.bottom + space
    }
  }
}

/**
 * Shows an element
 * @param {HTMLElement} element
 */
const showPopover = popover => {
  popover.removeAttribute('hidden')
}

/**
 * Hides an element
 * @param {HTMLElement} element
 */
const hidePopover = popover => {
  popover.setAttribute('hidden', true)
}

/**
 * Get keyboard focusable items within an element
 * @param {HTMLElement} element
 */
const getKeyboardFocusableElements = element => {
  return [...element.querySelectorAll(
    'a, button, input, textarea, select, [tabindex]:not([tabindex="-1"])'
  )]
}

// ========================
// Execution
// ========================
// Positions popover
popoverTriggers.forEach(popoverTrigger => {
  const popover = getPopover(popoverTrigger) || createPopover(popoverTrigger)
  const popoverPosition = calculatePopoverPosition(popoverTrigger, popover)

  popover.style.top = `${popoverPosition.top}px`
  popover.style.left = `${popoverPosition.left}px`
  hidePopover(popover)
})

// Show or hide popover when user clicks on the trigger
document.addEventListener('click', event => {
  const popoverTrigger = event.target.closest('.popover-trigger')
  if (!popoverTrigger) return

  const popover = document.querySelector(`#${popoverTrigger.dataset.target}`)
  if (popover.hasAttribute('hidden')) {
    showPopover(popover)
  } else {
    hidePopover(popover)
  }
})

// Hides popover user clicks something other than trigger or popover
document.addEventListener('click', event => {
  if (!event.target.closest('.popover') && !event.target.closest('.popover-trigger')) {
    const popovers = [...document.querySelectorAll('.popover')]
    popovers.forEach(popover => hidePopover(popover))
  }
})

// Allows Tabbing into Popover
document.addEventListener('keydown', event => {
  const { key } = event
  if (key !== 'Tab') return
  if (event.shiftKey) return

  const popoverTrigger = event.target.closest('.popover-trigger')
  if (!popoverTrigger) return

  const popover = getPopover(popoverTrigger)
  const focusables = getKeyboardFocusableElements(popover)
  const shouldTabIntoPopover = !popover.hasAttribute('hidden') && focusables.length !== 0

  if (shouldTabIntoPopover) {
    event.preventDefault()
    focusables[0].focus()
  }
})

// Tabs out of popover
document.addEventListener('keydown', event => {
  const popover = event.target.closest('.popover')
  if (!popover) return
  if (event.key !== 'Tab') return

  const popoverTrigger = getPopoverTrigger(popover)
  const focusables = getKeyboardFocusableElements(popover)

  if (event.shiftKey && event.target === focusables[0]) {
    event.preventDefault()
    return popoverTrigger.focus()
  }

  if (!event.shiftKey && event.target === focusables[focusables.length - 1]) {
    return popoverTrigger.focus()
  }
})

// Escape to close popover
document.addEventListener('keydown', event => {
  const { key } = event
  if (key !== 'Escape') return

  const popover = event.target.closest('.popover')
  if (!popover) return

  hidePopover(popover)
  const popoverTrigger = getPopoverTrigger(popover)
  popoverTrigger.focus()
})
